<?php
namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\BuildingModel;
use Illuminate\Http\Request;

class BuildingController extends Controller
{
    public function index(Request $request)
    {
        try {
            // Query dengan relasi office
            $query = BuildingModel::with(['office:id,office_name,office_short_name']);

            // Filter berdasarkan pencarian
            if (! empty($request->search)) {
                $query->where(function ($q) use ($request) {
                    $q->where('building_name', 'like', '%' . $request->search . '%')
                        ->orWhere('address', 'like', '%' . $request->search . '%')
                        ->orWhereHas('office', function ($q) use ($request) {
                            $q->where('office_name', 'like', '%' . $request->search . '%')
                                ->orWhere('office_short_name', 'like', '%' . $request->search . '%');
                        });
                });
            }

            // Pagination dan pengurutan berdasarkan ID terbaru
            $itemsPerPage = is_numeric($request->itemsPerPage) && $request->itemsPerPage > 0 ? (int) $request->itemsPerPage : 10;
            $buildings    = $query->orderBy('id', 'DESC')->paginate($itemsPerPage);

            // Ubah status menjadi integer agar tidak terjadi kesalahan di frontend
            $buildings->getCollection()->transform(function ($building) {
                $building->status = (int) $building->status;
                return $building;
            });

            // Kembalikan response dalam format yang mudah dibaca oleh frontend
            return response()->json([
                'data'         => $buildings->items(),
                'current_page' => $buildings->currentPage(),
                'last_page'    => $buildings->lastPage(),
                'total'        => $buildings->total(),
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error fetching building data.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function getListBuildings(Request $request)
    {
        try {
            // Query dengan relasi office (opsional)
            $query = BuildingModel::select('id', 'building_name');

            // Filter berdasarkan office_id jika diberikan
            if (! empty($request->office_id)) {
                $query->where('office_id', $request->office_id);
            }

            // Filter berdasarkan pencarian jika diberikan
            if (! empty($request->search)) {
                $query->where('building_name', 'like', '%' . $request->search . '%');
            }

            // Ambil data
            $buildings = $query->get();

            // Kembalikan data
            return response()->json([
                'data' => $buildings,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error fetching building list.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function store(Request $request)
    {
        $request->validate([
            'office_id'     => 'required|exists:tbl_office,id',
            'building_name' => 'required|string|max:255',
            'address'       => 'required|string',
            'status'        => 'required|in:0,1',
        ]);

        try {
            $data           = $request->all();
            $data['status'] = (int) $request->status; // Paksa status menjadi integer

            $building = BuildingModel::create($data);
            return response()->json($building->load('office:id,office_name,office_short_name'), 201);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error creating building.', 'error' => $e->getMessage()], 500);
        }
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'office_id'     => 'required|exists:tbl_office,id',
            'building_name' => 'required|string|max:255',
            'address'       => 'required|string',
            'status'        => 'required|in:0,1',
        ]);

        try {
            $building = BuildingModel::findOrFail($id);

            $data           = $request->all();
            $data['status'] = (int) $request->status; // Paksa status menjadi integer

            $building->update($data);
            return response()->json($building->load('office:id,office_name,office_short_name'), 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error updating building.', 'error' => $e->getMessage()], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $building = BuildingModel::findOrFail($id);

            // Ubah status menjadi 0 (menonaktifkan)
            $building->update([
                'status'     => 0,
                'updated_at' => now('Asia/Jakarta'),
            ]);

            return response()->json(['message' => 'Building deactivated successfully.'], 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error deactivating building.', 'error' => $e->getMessage()], 500);
        }
    }
}
