<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\CCTVModel;

class CCTVMobileController extends Controller
{
    /**
     * Fetch all CCTV data for mobile use
     */
    public function index(Request $request)
    {
        try {
            // Query untuk mendapatkan CCTV, Building, dan Office
            $query = CCTVModel::with([
                'building:id,building_name,office_id', // Include Building
                'building.office:id,office_name,office_short_name' // Include Office
            ]);

            // Filter berdasarkan pencarian
            if ($request->has('search') && !empty($request->search)) {
                $query->where('camera_name', 'like', '%' . $request->search . '%')
                    ->orWhere('camera_id', 'like', '%' . $request->search . '%')
                    ->orWhere('ip_address', 'like', '%' . $request->search . '%')
                    ->orWhere('ip_address_intranet', 'like', '%' . $request->search . '%')
                    ->orWhereHas('building', function ($q) use ($request) {
                        $q->where('building_name', 'like', '%' . $request->search . '%')
                            ->orWhereHas('office', function ($q) use ($request) {
                                $q->where('office_name', 'like', '%' . $request->search . '%')
                                    ->orWhere('office_short_name', 'like', '%' . $request->search . '%');
                            });
                    });
            }

            // Ambil semua data tanpa pagination
            $cctvs = $query->orderBy('id', 'desc')->get();

            return response()->json($cctvs, 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error fetching CCTV data.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get CCTVs by Office ID
     */
    public function getCCTVsByOffice($officeId)
    {
        try {
            $cctvs = CCTVModel::whereHas('building', function ($query) use ($officeId) {
                $query->where('office_id', $officeId);
            })->with([
                'building:id,building_name,office_id',
                'building.office:id,office_name,office_short_name'
            ])
                ->orderBy('camera_name', 'asc')
                ->get();

            return response()->json($cctvs, 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error fetching CCTV data by office.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Show a specific CCTV
     */
    public function show($id)
    {
        try {
            $cctv = CCTVModel::with([
                'building:id,building_name,office_id',
                'building.office:id,office_name,office_short_name'
            ])->findOrFail($id);

            return response()->json($cctv, 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'CCTV not found.', 'error' => $e->getMessage()], 404);
        }
    }

    /**
     * Create a new CCTV
     */
    public function store(Request $request)
    {
        $request->validate([
            'camera_id' => 'required|string|max:255|unique:tbl_cctv,camera_id',
            'camera_name' => 'required|string|max:255',
            'ip_address' => 'required|ip',
            'ip_address_intranet' => 'nullable|ip',
            'resolution' => 'nullable|string|max:50',
            'building_id' => 'required|exists:tbl_buildings,id',
            'status' => 'required|boolean',
        ]);

        try {
            $cctv = CCTVModel::create($request->all());
            return response()->json($cctv->load('building:id,building_name'), 201);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error creating CCTV.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Update an existing CCTV
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'camera_id' => 'required|string|max:255|unique:tbl_cctv,camera_id,' . $id,
            'camera_name' => 'required|string|max:255',
            'ip_address' => 'required|ip',
            'ip_address_intranet' => 'nullable|ip',
            'resolution' => 'nullable|string|max:50',
            'building_id' => 'required|exists:tbl_buildings,id',
            'status' => 'required|boolean',
        ]);

        try {
            $cctv = CCTVModel::findOrFail($id);
            $cctv->update($request->all());

            return response()->json($cctv->load('building:id,building_name'), 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error updating CCTV.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Delete a CCTV
     */
    public function destroy($id)
    {
        try {
            $cctv = CCTVModel::findOrFail($id);
            $cctv->delete();

            return response()->json(['message' => 'CCTV deleted successfully.'], 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error deleting CCTV.', 'error' => $e->getMessage()], 500);
        }
    }
}
