<?php
namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\CCTVModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CctvController extends Controller
{
    /**
     * Display a listing of the resource with pagination.
     */
    public function index(Request $request)
    {
        $query = CCTVModel::with('building.office'); // Include building and office data

        // Pencarian
        if ($request->has('search') && ! empty($request->search)) {
            $query->where(function ($q) use ($request) {
                $q->where('camera_name', 'like', '%' . $request->search . '%')
                    ->orWhere('ip_address', 'like', '%' . $request->search . '%')
                    ->orWhere('ip_address_intranet', 'like', '%' . $request->search . '%');
            });
        }

        // Validasi Input
        $perPage = intval($request->input('itemsPerPage', 10));
        $perPage = $perPage > 0 ? $perPage : 10;

        // Ambil Data dengan Paginasi
        $cctvs = $query->orderBy('id', 'desc')->paginate($perPage);

        return response()->json([
            'data'         => $cctvs->items(),
            'current_page' => $cctvs->currentPage(),
            'last_page'    => $cctvs->lastPage(),
            'total'        => $cctvs->total(),
            'per_page'     => $cctvs->perPage(),
        ]);
    }

    /**
     * Retrieve all CCTV data without pagination.
     */
    public function all()
    {
        $cctvs = CCTVModel::with('building.office')->get(); // Include building and office data
        return response()->json($cctvs);
    }

    public function getListCctvStreams(Request $request)
    {
        try {
            // Validasi input
            $request->validate([
                'category'     => 'in:internet,intranet', // Hanya menerima 'internet' atau 'intranet'
                'camera_id'    => 'integer|exists:tbl_cctv,id',
                'office_id'    => 'integer|exists:tbl_office,id',
                'building_id'  => 'integer|exists:tbl_building,id',
                'itemsPerPage' => 'integer|min:1|max:100', // Batas pagination
            ]);

                                                                 // Tentukan kategori (internet/intranet)
            $category = $request->input('category', 'internet'); // Default 'internet'

            // Mulai query
            $query = CCTVModel::with('building.office');

            // Filter berdasarkan kategori
            if ('intranet' === $category) {
                $query->whereNotNull('ip_address_intranet');
            } else {
                $query->where('status', 1);
            }

            // Filter berdasarkan camera_id jika ada
            if ($request->filled('camera_id')) {
                $query->where('id', $request->input('camera_id'));
            }

            // Filter berdasarkan office_id jika ada
            if ($request->filled('office_id')) {
                $query->whereHas('building.office', function ($q) use ($request) {
                    $q->where('id', $request->input('office_id'));
                });
            }

            // Filter berdasarkan building_id jika ada
            if ($request->filled('building_id')) {
                $query->whereHas('building', function ($q) use ($request) {
                    $q->where('id', $request->input('building_id'));
                });
            }

            $query->orderBy('camera_id', 'asc');

            // Ambil jumlah data per halaman, default 8
            $itemsPerPage = $request->input('itemsPerPage', 8);

            // Ambil data dengan pagination
            $cctvs = $query->paginate($itemsPerPage);

            // Kembalikan response dengan meta data pagination
            return response()->json([
                'data'         => $cctvs->items(),
                'currentPage'  => $cctvs->currentPage(),
                'lastPage'     => $cctvs->lastPage(),
                'itemsPerPage' => $cctvs->perPage(),
                'total'        => $cctvs->total(),
            ], 200);
        } catch (\Exception $e) {
            // Tangkap error dan kembalikan response
            return response()->json([
                'message' => 'Error fetching CCTV streams.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    // Untuk list option pada single stream
    public function getListAllCctvStreams(Request $request)
    {
        try {
            // Validasi input
            $request->validate([
                'category'    => 'in:internet,intranet', // Hanya menerima 'internet' atau 'intranet'
                'camera_id'   => 'integer|exists:tbl_cctv,id',
                'office_id'   => 'integer|exists:tbl_office,id',
                'building_id' => 'integer|exists:tbl_building,id',
            ]);

            $category = $request->input('category', 'internet'); // Default 'internet'

            $query = CCTVModel::with([
                'building'        => function ($query) {
                    $query->select('id', 'building_name', 'office_id'); // Menyertakan office_id untuk memetakan relasi
                },
                'building.office' => function ($query) {
                    $query->select('id', 'office_name'); // Pilih hanya kolom yang diinginkan dari office
                },
            ]);

            // Filter berdasarkan kategori
            if ('intranet' === $category) {
                $query->whereNotNull('ip_address_intranet');
            } else {
                $query->where('status', 1);
            }

            // Filter berdasarkan camera_id jika ada
            if ($request->filled('camera_id')) {
                $query->where('id', $request->input('camera_id'));
            }

            // Filter berdasarkan office_id jika ada
            if ($request->filled('office_id')) {
                $query->whereHas('building.office', function ($q) use ($request) {
                    $q->where('id', $request->input('office_id'));
                });
            }

            // Filter berdasarkan building_id jika ada
            if ($request->filled('building_id')) {
                $query->whereHas('building', function ($q) use ($request) {
                    $q->where('id', $request->input('building_id'));
                });
            }

            $cctvs = $query->get();

            return response()->json([
                'data' => $cctvs,
            ], 200);
        } catch (\Exception $e) {
            // Tangkap error dan kembalikan response
            return response()->json([
                'message' => 'Error fetching CCTV streams.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'camera_name'         => 'required|string|max:255',
            'ip_address'          => 'required|unique:tbl_cctv',
            'ip_address_intranet' => 'nullable',
            'resolution'          => 'nullable|string|max:255',
            'building_id'         => 'required|exists:tbl_building,id',
            'status'              => 'required|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $cctv = CCTVModel::create($validator->validated());
        return response()->json($cctv->load('building.office'), 201);
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $cctv = CCTVModel::with('building.office')->find($id);

        if (! $cctv) {
            return response()->json(['message' => 'CCTV not found'], 404);
        }

        return response()->json($cctv);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'camera_name'         => 'sometimes|required|string|max:255',
            'ip_address'          => 'sometimes|required|unique:tbl_cctv,ip_address,' . $id,
            'ip_address_intranet' => 'nullable',
            'resolution'          => 'nullable|string|max:255',
            'building_id'         => 'sometimes|required|exists:tbl_building,id',
            'status'              => 'required|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $cctv = CCTVModel::find($id);

        if (! $cctv) {
            return response()->json(['message' => 'CCTV not found'], 404);
        }

        $cctv->update($validator->validated());
        return response()->json($cctv->load('building.office'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $cctv = CCTVModel::find($id);

        if (! $cctv) {
            return response()->json(['message' => 'CCTV not found'], 404);
        }

        $cctv->delete();
        return response()->json(['message' => 'CCTV deleted successfully']);
    }
}