<?php
namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\CctvLiveMonitoringProjectModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CctvLiveMonitoringProjectController extends Controller
{
    /**
     * Display a listing of the resource with pagination.
     */
    public function index(Request $request)
    {
        $query = CctvLiveMonitoringProjectModel::query();

        // Pencarian
        if ($request->has('search') && ! empty($request->search)) {
            $query->where(function ($q) use ($request) {
                $q->where('camera_name', 'like', '%' . $request->search . '%')
                    ->orWhere('url', 'like', '%' . $request->search . '%');
            });
        }

        // Validasi Input
        $perPage = intval($request->input('itemsPerPage', 10));
        $perPage = $perPage > 0 ? $perPage : 10;

        // Ambil Data dengan Paginasi
        $cctvs = $query->orderBy('id', 'desc')->paginate($perPage);

        return response()->json([
            'data'         => $cctvs->items(),
            'current_page' => $cctvs->currentPage(),
            'last_page'    => $cctvs->lastPage(),
            'total'        => $cctvs->total(),
            'per_page'     => $cctvs->perPage(),
        ]);
    }

    /**
     * Retrieve all CCTV data without pagination.
     */
    public function all()
    {
        $cctvs = CctvLiveMonitoringProjectModel::all();
        return response()->json($cctvs);
    }

    /**
     * Store a newly created resource.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'camera_name' => 'required|string|max:255',
            'url'         => 'required|string',
            'status'      => 'required|in:0,1',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $cctv = CctvLiveMonitoringProjectModel::create($validator->validated());
        return response()->json($cctv, 201);
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $cctv = CctvLiveMonitoringProjectModel::find($id);

        if (! $cctv) {
            return response()->json(['message' => 'CCTV not found'], 404);
        }

        return response()->json($cctv);
    }

    /**
     * Update the specified resource.
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'camera_name' => 'sometimes|required|string|max:255',
            'url'         => 'sometimes|required|string',
            'status'      => 'required|in:0,1',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $cctv = CctvLiveMonitoringProjectModel::find($id);

        if (! $cctv) {
            return response()->json(['message' => 'CCTV not found'], 404);
        }

        $cctv->update($validator->validated());
        return response()->json($cctv);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $cctv = CctvLiveMonitoringProjectModel::find($id);

        if (! $cctv) {
            return response()->json(['message' => 'CCTV not found'], 404);
        }

        $cctv->delete();
        return response()->json(['message' => 'CCTV deleted successfully']);
    }
}
