<?php
namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class PlateNumberMobileController extends Controller
{

    public function getListAllPlateNumber(Request $request)
    {
        $apiKey = $request->header('x-api-key');

        // Validasi API Key
        if (! $apiKey) {
            return response()->json(['error' => 'API Key is required'], 400);
        }

        if (config('app.mobile_api_key') !== $apiKey) {
            return response()->json(['error' => 'Invalid API Key'], 401);
        }

        // Ambil parameter opsional dari request
        $userId      = $request->input('user_id');
        $plateNumber = $request->input('plate_number');

        // Query dasar untuk mengambil semua plat nomor dengan join ke tbl_user dan tbl_office
        $query = DB::table('tbl_plate_number')
            ->join('tbl_user', 'tbl_plate_number.user_id', '=', 'tbl_user.id')
            ->leftJoin('tbl_office', 'tbl_user.office_id', '=', 'tbl_office.id')
            ->select(
                'tbl_plate_number.id',
                'tbl_plate_number.plate_number',
                'tbl_plate_number.status',
                'tbl_plate_number.created_at',
                'tbl_plate_number.updated_at',
                'tbl_user.id as user_id',
                'tbl_user.username',
                'tbl_user.ai_label',
                'tbl_user.nama_lengkap',
                'tbl_office.office_name'
            )
            ->where('tbl_plate_number.status', 1);

        // Jika user_id diberikan, tambahkan kondisi ke query
        if (! empty($userId)) {
            $query->where('tbl_plate_number.user_id', $userId);
        }

        // Jika plate_number diberikan, tambahkan kondisi ke query
        if (! empty($plateNumber)) {
            $query->where('tbl_plate_number.plate_number', 'LIKE', "%{$plateNumber}%");
        }

        // Eksekusi query dan ambil hasilnya
        $plateNumbers = $query->get();

        return response()->json([
            'message' => 'Plate numbers fetched successfully.',
            'data'    => $plateNumbers,
        ], 200);
    }

    /**
     * Add Plate Number for Mobile Users.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function addPlateNumber(Request $request)
    {
        $apiKey = $request->header('x-api-key');

        // Validate API Key
        if (! $apiKey) {
            return response()->json(['error' => 'API Key is required'], 400);
        }

        if (config('app.mobile_api_key') !== $apiKey) {
            return response()->json(['error' => 'Invalid API Key'], 401);
        }

        // Validate input
        $validator = Validator::make($request->all(), [
            'user_id'      => 'required|exists:tbl_user,id',
            'plate_number' => 'required|string|max:20',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 422);
        }

        // Check if plate number exists with status = 1
        $existingPlate = DB::table('tbl_plate_number')
            ->where('plate_number', $request->input('plate_number'))
            ->where('status', 1)
            ->first();

        if ($existingPlate) {
            return response()->json(['error' => 'Plate number is already in use'], 422);
        }

        // Insert plate number
        DB::table('tbl_plate_number')->insert([
            'user_id'      => $request->input('user_id'),
            'plate_number' => $request->input('plate_number'),
            'status'       => 1,
            'created_at'   => now('Asia/Jakarta'),
            'updated_at'   => now('Asia/Jakarta'),
        ]);

        return response()->json([
            'message' => 'Plate number successfully added.',
        ], 201);
    }

    /**
     * Update Plate Number for Mobile Users.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updatePlateNumber(Request $request)
    {
        $apiKey = $request->header('x-api-key');

        // Validate API Key
        if (! $apiKey) {
            return response()->json(['error' => 'API Key is required'], 400);
        }

        if (config('app.mobile_api_key') !== $apiKey) {
            return response()->json(['error' => 'Invalid API Key'], 401);
        }

        // Validate input
        $validator = Validator::make($request->all(), [
            'id'           => 'required|exists:tbl_plate_number,id',
            'plate_number' => 'required|string|max:20',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 422);
        }

        // Check if plate number exists with status = 1 (excluding current id)
        $existingPlate = DB::table('tbl_plate_number')
            ->where('plate_number', $request->input('plate_number'))
            ->where('status', 1)
            ->where('id', '!=', $request->input('id'))
            ->first();

        if ($existingPlate) {
            return response()->json(['error' => 'Plate number is already in use'], 422);
        }

        // Update plate number
        DB::table('tbl_plate_number')->where('id', $request->input('id'))->update([
            'plate_number' => $request->input('plate_number'),
            'updated_at'   => now('Asia/Jakarta'),
        ]);

        return response()->json([
            'message' => 'Plate number successfully updated.',
        ], 200);
    }

    /**
     * Delete Plate Number for Mobile Users (Soft Delete).
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function deletePlateNumber(Request $request)
    {
        $apiKey = $request->header('x-api-key');

        // Validate API Key
        if (! $apiKey) {
            return response()->json(['error' => 'API Key is required'], 400);
        }

        if (config('app.mobile_api_key') !== $apiKey) {
            return response()->json(['error' => 'Invalid API Key'], 401);
        }

        // Validate input
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:tbl_plate_number,id',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 422);
        }

        // Soft delete plate number (update status to 0)
        DB::table('tbl_plate_number')->where('id', $request->input('id'))->update([
            'status'     => 0,
            'updated_at' => now('Asia/Jakarta'),
        ]);

        return response()->json([
            'message' => 'Plate number successfully deleted.',
        ], 200);
    }

    /**
     * Get List of Plate Numbers by User ID.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getPlateNumbersByUser(Request $request)
    {
        $apiKey = $request->header('x-api-key');

        // Validate API Key
        if (! $apiKey) {
            return response()->json(['error' => 'API Key is required'], 400);
        }

        if (config('app.mobile_api_key') !== $apiKey) {
            return response()->json(['error' => 'Invalid API Key'], 401);
        }

        // Validate input
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:tbl_user,id',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 422);
        }

        // Fetch plate numbers
        $plateNumbers = DB::table('tbl_plate_number')
            ->where('user_id', $request->input('user_id'))
            ->where('status', 1)
            ->get();

        return response()->json([
            'message' => 'Plate numbers fetched successfully.',
            'data'    => $plateNumbers,
        ], 200);
    }

    public function validatePlateNumber(Request $request)
    {
        $apiKey = $request->header('x-api-key');

        // Validate API Key
        if (! $apiKey) {
            return response()->json(['error' => 'API Key is required'], 400);
        }

        if (config('app.mobile_api_key') !== $apiKey) {
            return response()->json(['error' => 'Invalid API Key'], 401);
        }

        // Clean and sanitize input
        $rawPlateNumber   = $request->input('plate_number');
        $cleanPlateNumber = preg_replace('/[^a-zA-Z0-9]/', '', $rawPlateNumber); // Hanya huruf dan angka
        $cleanPlateNumber = str_replace(' ', '', $cleanPlateNumber);             // Hapus spasi

        // Validate input
        $validator = Validator::make(['plate_number' => $cleanPlateNumber], [
            'plate_number' => 'required|string|max:20',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 422);
        }

        // Clean data from database for comparison
        $plateNumbers = DB::table('tbl_plate_number')
            ->join('tbl_user', 'tbl_plate_number.user_id', '=', 'tbl_user.id')
            ->select(
                'tbl_plate_number.id as plate_id',
                DB::raw("REPLACE(REGEXP_REPLACE(tbl_plate_number.plate_number, '[^a-zA-Z0-9]', ''), ' ', '') as cleaned_plate_number"),
                'tbl_plate_number.status',
                'tbl_user.id as user_id',
                'tbl_user.nama_lengkap',
                'tbl_user.email',
                'tbl_user.level_id',
                'tbl_user.jabatan',
                'tbl_user.nip'
            )
            ->where('tbl_plate_number.status', '1')
            ->get();

        // Compare sanitized input with sanitized database values
        $plateNumber = $plateNumbers->first(function ($plate) use ($cleanPlateNumber) {
            return $plate->cleaned_plate_number === $cleanPlateNumber;
        });

        if ($plateNumber) {
            return response()->json([
                'status'  => true,
                'message' => 'PLATE NUMBER IS VALID',
                'data'    => [
                    'user_id'      => $plateNumber->user_id,
                    'nama_lengkap' => $plateNumber->nama_lengkap,
                    'email'        => $plateNumber->email,
                    'level_id'     => $plateNumber->level_id,
                    'jabatan'      => $plateNumber->jabatan,
                    'nip'          => $plateNumber->nip,
                ],
            ], 200);
        } else {
            return response()->json([
                'status'  => false,
                'message' => 'PLATE NUMBER IS INVALID',
            ], 200);
        }
    }
     public function validatePlateNumberWithUser(Request $request)
    {
        $apiKey = $request->header('x-api-key');

        // Validate API Key
        if (! $apiKey) {
            return response()->json(['error' => 'API Key is required'], 400);
        }

        if (config('app.mobile_api_key') !== $apiKey) {
            return response()->json(['error' => 'Invalid API Key'], 401);
        }

        // Clean and sanitize input
        $rawPlateNumber   = $request->input('plate_number');
        $cleanPlateNumber = preg_replace('/[^a-zA-Z0-9]/', '', $rawPlateNumber); // Hanya huruf dan angka
        $cleanPlateNumber = str_replace(' ', '', $cleanPlateNumber);             // Hapus spasi

        // Validate input
        $validator = Validator::make(['plate_number' => $cleanPlateNumber], [
            'plate_number' => 'required|string|max:20',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 422);
        }

        // Clean data from database for comparison
        $plateNumbers = DB::table('tbl_plate_number')
            ->join('tbl_user', 'tbl_plate_number.user_id', '=', 'tbl_user.id')
            ->select(
                'tbl_plate_number.id as plate_id',
                'tbl_plate_number.plate_number',
                'tbl_plate_number.status',
                'tbl_user.id as user_id',
                'tbl_user.nama_lengkap',
                'tbl_user.email',
                'tbl_user.level_id',
                'tbl_user.jabatan',
                'tbl_user.nip',
                'tbl_user.ai_label'
            )
            ->where('tbl_plate_number.status', '1')
            ->whereRaw("REPLACE(REGEXP_REPLACE(tbl_plate_number.plate_number, '[^a-zA-Z0-9]', ''), ' ', '') LIKE ?", [$cleanPlateNumber . '%'])
            ->get();

        if ($plateNumbers->isEmpty()) {
            return response()->json([
                'status'  => false,
                'message' => 'PLATE NUMBER IS INVALID',
                'data'    => [],
            ], 200);
        }

        $data = $plateNumbers->map(function ($plate) {
            return [
                'plate_id'      => $plate->plate_id,
                'plate_number'  => $plate->plate_number,
                'user_id'       => $plate->user_id,
                'nama_lengkap'  => $plate->nama_lengkap,
                'email'         => $plate->email,
                'level_id'      => $plate->level_id,
                'jabatan'       => $plate->jabatan,
                'nip'           => $plate->nip,
                'ai_label'      => $plate->ai_label
            ];
        });

        return response()->json([
            'status'  => true,
            'message' => 'PLATE NUMBER IS VALID',
            'data'    => $data,
        ], 200);

        
    }

}