<?php
namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\UserLevelModel;
use Illuminate\Http\Request;

class RoleController extends Controller
{
    /**
     * Get all roles.
     */
    public function index()
    {
        try {
            $roles = UserLevelModel::where('status', "1")->get(); // Hanya ambil role aktif
            return response()->json($roles, 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error fetching roles.', 'error' => $e->getMessage()], 500);
        }
    }

    public function getRoleList()
    {
        try {
            $roles = UserLevelModel::where('status', "1")
                ->where('id', '>', 1) // Tambahkan kondisi untuk id > 1
                ->orderBy('level_name', 'asc')
                ->get(['id', 'level_name']);

            return response()->json([
                'success' => true,
                'data'    => $roles,
            ], 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error fetching role list.', 'error' => $e->getMessage()], 500);
        }
    }

    public function getRoleToUpdateUserList()
    {
        try {
            $roles = UserLevelModel::where('status', "1")
                ->whereIn('id', [2, 3, 4, 5, 7, 8, 9, 10]) // Gunakan whereIn untuk id yang diizinkan
                ->orderBy('level_name', 'asc')
                ->get(['id', 'level_name']);

            return response()->json([
                'success' => true,
                'data'    => $roles,
            ], 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error fetching role list.', 'error' => $e->getMessage()], 500);
        }
    }

    public function store(Request $request)
    {
        $request->validate([
            'level_name' => 'required|string|max:255',
            'status'     => 'required|boolean',
        ]);

        try {
            $role = UserLevelModel::create($request->all());
            return response()->json(['message' => 'Role created successfully.', 'data' => $role], 201);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error creating role.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get a single role by ID.
     */
    public function show($id)
    {
        try {
            $role = UserLevelModel::findOrFail($id);
            return response()->json($role, 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Role not found.', 'error' => $e->getMessage()], 404);
        }
    }

    /**
     * Update an existing role.
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'level_name' => 'required|string|max:255',
            'status'     => 'required|boolean',
        ]);

        try {
            $role = UserLevelModel::findOrFail($id);
            $role->update($request->all());
            return response()->json(['message' => 'Role updated successfully.', 'data' => $role], 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error updating role.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Delete a role by ID (soft delete by changing status).
     */
    public function destroy($id)
    {
        try {
            $role = UserLevelModel::findOrFail($id);
            $role->update(['status' => 0]); // Nonaktifkan role
            return response()->json(['message' => 'Role deactivated successfully.'], 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error deactivating role.', 'error' => $e->getMessage()], 500);
        }
    }
}