<?php
namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\UserLevelModel;
use App\Models\UserModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Http;

class UserController extends Controller
{
    public function index(Request $request)
    {
        $query = UserModel::with(['userLevel', 'office'])->whereIn('level_id', [2, 3, 4, 5, 7, 8, 9, 10]);

        // Filter berdasarkan pencarian
        if ($request->has('search') && ! empty($request->search)) {
            $query->where(function ($q) use ($request) {
                $q->where('nama_lengkap', 'like', '%' . $request->search . '%')
                    ->orWhere('username', 'like', '%' . $request->search . '%')
                    ->orWhere('email', 'like', '%' . $request->search . '%')
                    ->orWhere('nip', 'like', '%' . $request->search . '%')
                    ->orWhere('ai_label', 'like', '%' . $request->search . '%')
                    ->orWhere('jabatan', 'like', '%' . $request->search . '%');
            });
        }

        // Filter berdasarkan office_id
        if ($request->has('office_id') && ! empty($request->office_id)) {
            $query->where('office_id', $request->office_id); // Filter pengguna berdasarkan office
        }

        if ($request->has('role_id') && ! empty($request->role_id)) {
            $query->where('level_id', $request->role_id); // Filter pengguna berdasarkan office
        }

        $itemsPerPage = $request->itemsPerPage ?? 10; // Default items per page
        $users        = $query->orderBy('id', 'DESC')->paginate($itemsPerPage);

        // Transform response untuk menyertakan data office
        $users->getCollection()->transform(function ($user) {
            $user->office_name = $user->office->office_name ?? null; // Sertakan nama office jika ada
            return $user;
        });

        return response()->json($users);
    }

    public function getUserList()
    {
        try {
            $data = UserModel::where('status', operator: "1")
                ->orderBy('id', 'asc')
                ->get(['id', 'username', 'nama_lengkap', 'ai_label']);

            return response()->json([
                'success' => true,
                'data'    => $data,
            ], 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error fetching office list.', 'error' => $e->getMessage()], 500);
        }
    }

    public function show($id)
    {
        // Tambahkan relasi 'office' ke dalam with
        $user = UserModel::with(['userLevel', 'office'])->find($id);

        if (is_null($user)) {
            return response()->json(['message' => 'User not found'], 404);
        }

        // Sertakan nama office jika ada
        $user->office_name = $user->office->office_name ?? null;

        return response()->json($user);
    }

    public function store(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'username'     => 'required|string|max:255|unique:tbl_user',
                'password'     => 'required|string',
                'office_id'    => 'required',
                'nama_lengkap' => 'required|string|max:255',
                'level_id'     => 'required',
                'nip'          => 'required|string|max:20|unique:tbl_user',
                'jabatan'      => 'required|string',
                'email'        => 'required|email|string|max:255|unique:tbl_user',
                'no_hp'        => 'nullable|string|max:20',
            ]);

            $validatedData['password'] = Hash::make($validatedData['password']);

            // Simpan user baru
            $user = UserModel::create($validatedData);

            // Format ID jadi 4 digit
            $formattedId = str_pad($user->id, 4, '0', STR_PAD_LEFT);

            // Ambil level name dari tabel user_level
            $levelName = UserLevelModel::where('id', $validatedData['level_id'])->value('level_name');

            // Ambil hanya kata pertama dari nama_lengkap
            $firstName = explode(' ', trim($user->nama_lengkap))[0];

            // Format AI Label
            $aiLabel = "{$formattedId}-{$levelName}-{$firstName}-{$user->nip}";

            // Update AI Label di database
            $user->update(['ai_label' => $aiLabel]);

            // **Hit API untuk menambahkan label baru**
            $apiResponse = Http::withHeaders([
                'x-api-key' => '9115e6b4-f105-4d78-8bf3-718768d4eb48',
            ])->post('https://api-cpf.bantungoding.com/api/v1/recognition/subjects', [
                'subject' => $aiLabel,
            ]);

            if ($apiResponse->failed()) {
                return response()->json(['error' => 'Failed to add label in external API.'], $apiResponse->status());
            }

            return response()->json($user, 201);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $validatedData = $request->validate([
                'username'     => 'sometimes|required|string|max:255|unique:tbl_user,username,' . $id,
                'nama_lengkap' => 'sometimes|required|string|max:255',
                'office_id'    => 'sometimes|required',
                'level_id'     => 'sometimes|required',
                'email'        => 'nullable|string|max:255',
                'no_hp'        => 'nullable|string|max:20',
                'nip'          => 'required|string|max:20|unique:tbl_user,nip,' . $id,
                'jabatan'      => 'required|string',
            ]);

            $user = UserModel::find($id);
            if (is_null($user)) {
                return response()->json(['message' => 'User not found'], 404);
            }

            $oldAiLabel = $user->ai_label; // Simpan label lama sebelum diupdate
            $user->update($validatedData);

            // Format ID jadi 4 digit
            $formattedId = str_pad($user->id, 4, '0', STR_PAD_LEFT);

            // Ambil level name dari tabel user_level
            $levelName = UserLevelModel::where('id', $user->level_id)->value('level_name');

            // Ambil hanya kata pertama dari nama_lengkap
            $firstName = explode(' ', trim($user->nama_lengkap))[0];

            // Format AI Label baru
            $newAiLabel = "{$formattedId}-{$levelName}-{$firstName}-{$user->nip}";

            // **Jika AI Label sudah ada, lakukan update di API**
            if (! empty($oldAiLabel)) {
                $apiResponse = Http::withHeaders([
                    'x-api-key' => '9115e6b4-f105-4d78-8bf3-718768d4eb48',
                ])->put("https://api-cpf.bantungoding.com/api/v1/recognition/subjects/{$oldAiLabel}", [
                    'subject' => $newAiLabel,
                ]);

                if ($apiResponse->failed()) {
                    return response()->json(['error' => 'Failed to update label in external API.'], $apiResponse->status());
                }
            } else {
                // Jika belum ada, tambahkan label baru ke API
                $apiResponse = Http::withHeaders([
                    'x-api-key' => '9115e6b4-f105-4d78-8bf3-718768d4eb48',
                ])->post('https://api-cpf.bantungoding.com/api/v1/recognition/subjects', [
                    'subject' => $newAiLabel,
                ]);

                if ($apiResponse->failed()) {
                    return response()->json(['error' => 'Failed to add label in external API.'], $apiResponse->status());
                }
            }

            // Update AI Label di database
            $user->update(['ai_label' => $newAiLabel]);

            return response()->json($user);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        }
    }

    // Create a new user
    // public function store(Request $request)
    // {
    //     try {
    //         $validatedData = $request->validate([
    //             'username'     => 'required|string|max:255|unique:tbl_user',
    //             'password'     => 'required|string',
    //             'office_id'    => 'required',
    //             'nama_lengkap' => 'required|string|max:255',
    //             'level_id'     => 'required',
    //             'nip'          => 'required|string|max:20|unique:tbl_user',
    //             'jabatan'      => 'required|string',
    //             'email'        => 'required|email|string|max:255|unique:tbl_user',
    //             'no_hp'        => 'nullable|string|max:20',
    //             'ai_label'     => 'nullable|string|max:50',
    //         ], [
    //             'username.required'     => 'Username tidak boleh kosong.',
    //             'username.unique'       => 'Username telah digunakan.',
    //             'email.required'        => 'Email tidak boleh kosong.',
    //             'email.unique'          => 'Email telah digunakan.',
    //             'email.email'           => 'Email tidak valid !',
    //             'nip.required'          => 'NIP tidak boleh kosong.',
    //             'nip.unique'            => 'NIP telah terdaftar.',
    //             'jabatan.required'      => 'Jabatan tidak boleh kosong.',
    //             'password.required'     => 'Password tidak boleh kosong.',
    //             'nama_lengkap.required' => 'Nama lengkap tidak boleh kosong.',
    //             'office_id.required'    => 'Office tidak boleh kosong.',
    //             'level_id.required'     => 'Role user tidak boleh kosong.',
    //             'level_id.in'           => 'Invalid role !',
    //             'no_hp.max'             => 'Phone number must not exceed 16 characters.',
    //         ]);

    //         $validatedData['level_id'] = $request->input('level_id', 1);
    //         $validatedData['password'] = Hash::make($validatedData['password']);
    //         $user                      = UserModel::create($validatedData);

    //         return response()->json($user, 201);
    //     } catch (\Illuminate\Validation\ValidationException $e) {
    //         return response()->json(['errors' => $e->errors()], 422);
    //     }
    // }

    // // Update a specific user by id
    // public function update(Request $request, $id)
    // {
    //     try {
    //         $validatedData = $request->validate([
    //             'username'     => 'sometimes|required|string|max:255|unique:tbl_user,username,' . $id,
    //             'nama_lengkap' => 'sometimes|required|string|max:255',
    //             'office_id'    => 'sometimes|required',
    //             'level_id'     => 'sometimes|required',
    //             'email'        => 'nullable|string|max:255',
    //             'no_hp'        => 'nullable|string|max:20',
    //             'nip'          => 'required|string|max:20|unique:tbl_user,nip,' . $id,
    //             'jabatan'      => 'required|string',
    //             'ai_label'     => 'nullable|string|max:50',
    //         ], [
    //             'username.required'     => 'Username tidak boleh kosong.',
    //             'username.unique'       => 'Username telah digunakan.',
    //             'nip.required'          => 'NIP tidak boleh kosong.',
    //             'nip.unique'            => 'NIP telah digunakan.',
    //             'jabatan.required'      => 'Jabatan tidak boleh kosong.',
    //             'nama_lengkap.required' => 'Full name tidak boleh kosong.',
    //             'office_id.required'    => 'Role user tidak boleh kosong.',
    //             'level_id.required'     => 'Role user tidak boleh kosong.',
    //             'no_hp.max'             => 'Phone number must not exceed 20 characters.',
    //             'nip.max'               => 'NIP tidak boleh lebih dari 20 karakter.',
    //             'ai_label.max'          => 'LABEL AI tidak boleh lebih dari 50 karakter.',
    //         ]);

    //         $user = UserModel::find($id);
    //         if (is_null($user)) {
    //             return response()->json(['message' => 'User not found'], 404);
    //         }

    //         $user->update($validatedData);
    //         return response()->json($user);
    //     } catch (\Illuminate\Validation\ValidationException $e) {
    //         return response()->json(['errors' => $e->errors()], 422);
    //     }
    // }

    // Update profile
    public function changeProfile(Request $request, $id)
    {
        try {
            $validatedData = $request->validate([
                'nama_lengkap' => 'required|string|max:255',
                'email'        => 'nullable|string|email|max:255',
                'no_hp'        => 'nullable|string|max:15',
            ], [
                'nama_lengkap.required' => 'Nama lengkap tidak boleh kosong !',
                'email.email'           => 'Email tidak valid !',
                'no_hp.max'             => 'No. Handphone tidak boleh lebih dari 15 karakter.',
            ]);

            $user = UserModel::find($id);
            if (is_null($user)) {
                return response()->json(['message' => 'User not found'], 404);
            }

            $user->update($validatedData);

            return response()->json($user);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        }
    }

    // Change password
    public function changePassword(Request $request, $id)
    {
        $validatedData = $request->validate([
            'new_password'              => 'required|string|min:8',
            'new_password_confirmation' => 'required|string|min:8|same:new_password',
        ], [
            'new_password.required'              => 'Password baru password harus diisi.',
            'new_password.min'                   => 'Konfirmasi password password minimal 8 karakter.',
            'new_password_confirmation.required' => 'Konfirmasi password harus diisi.',
            'new_password_confirmation.min'      => 'Konfirmasi password minimal 8 karakter.',
            'new_password_confirmation.same'     => 'Konfirmasi password tidak cocok !',
        ]);

        $user = UserModel::find($id);
        if (is_null($user)) {
            return response()->json(['message' => 'Terjadi kesalahan, Data tidak valid !', 'error' => true], 404);
        }

        $user->update(['password' => Hash::make($validatedData['new_password'])]);

        return response()->json(['message' => 'Password updated successfully', 'error' => false]);
    }

    public function changePasswordUser(Request $request, $id)
    {
        try {
            $validatedData = $request->validate([
                'old_password'              => 'required|string|min:8',
                'new_password'              => 'required|string|min:8',
                'new_password_confirmation' => 'required|string|min:8|same:new_password',
            ], [
                'old_password.required'              => 'Password lama harus diisi.',
                'old_password.min'                   => 'Password lama minimal 8 karakter.',
                'new_password.required'              => 'Password baru harus diisi.',
                'new_password.min'                   => 'Password baru minimal 8 karakter.',
                'new_password_confirmation.required' => 'Konfirmasi password harus diisi.',
                'new_password_confirmation.min'      => 'Konfirmasi password minimal 8 karakter.',
                'new_password_confirmation.same'     => 'Konfirmasi password tidak cocok!',
            ]);

            $user = UserModel::find($id);
            if (is_null($user)) {
                return response()->json(['message' => 'Terjadi kesalahan, data tidak valid!', 'error' => true], 404);
            }

            if (! Hash::check($validatedData['old_password'], $user->password)) {
                return response()->json(['message' => 'Password lama tidak cocok!', 'error' => true], 400);
            }

            $user->update(['password' => Hash::make($validatedData['new_password'])]);

            return response()->json(['message' => 'Password berhasil diperbarui', 'error' => false]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        }
    }

    // Deactivate account
    public function deactivateAccount($id)
    {
        $user = UserModel::find($id);
        if (is_null($user)) {
            return response()->json(['message' => 'User not found'], 404);
        }

        $user->update(['status' => '0']);

        return response()->json(['message' => 'Account deactivated successfully']);
    }

    // Activate account
    public function activateAccount($id)
    {
        $user = UserModel::find($id);
        if (is_null($user)) {
            return response()->json(['message' => 'User not found'], 404);
        }

        $user->update(['status' => '1']);

        return response()->json(['message' => 'Account activated successfully']);
    }

    // Get all users sorted by name ASC for list options
    public function listOptions()
    {
        $users = UserModel::whereIn('level_id', [2, 3, 4, 5, 7, 8, 9, 10])
            ->where('status', '1')
            ->orderBy('nama_lengkap', 'ASC')
            ->get(['id', 'nama_lengkap']);

        return response()->json($users);
    }

    // Delete a specific user by id
    public function destroy($id)
    {
        $user = UserModel::find($id);
        if (is_null($user)) {
            return response()->json(['message' => 'User not found'], 404);
        }

        $user->delete();

        return response()->json(['message' => 'User deleted successfully']);
    }
}
