<?php
namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;

class UserMobileController extends Controller
{
    /**
     * Update user data via Mobile API.
     */
    public function updateUser(Request $request)
    {
        $apiKey = $request->header('x-api-key');

        // Validasi API Key
        if (! $apiKey) {
            return response()->json(['error' => 'API Key is required'], 400);
        }

        if (config('app.mobile_api_key') !== $apiKey) {
            return response()->json(['error' => 'Invalid API Key'], 401);
        }

        $validatedData = $request->validate([
            'id'           => 'required|integer',
            'username'     => 'sometimes|required|string|max:255|unique:tbl_user,username,' . $request->id,
            'nama_lengkap' => 'sometimes|required|string|max:255',
            'email'        => 'nullable|string|max:255',
            'no_hp'        => 'nullable|string|max:20',
            'nip'          => 'required|string|max:20|unique:tbl_user,nip,' . $request->id,
            'jabatan'      => 'required|string',
            'office_id'    => 'sometimes|required|integer',
        ]);

        // Ambil user dari database
        $user = DB::table('tbl_user')->where('id', $request->id)->first();

        if (! $user) {
            return response()->json(['error' => 'User not found'], 404);
        }

        $oldAiLabel = $user->ai_label; // Simpan AI Label lama sebelum diupdate

        // Format ID jadi 4 digit
        $formattedId = str_pad($user->id, 4, '0', STR_PAD_LEFT);

        // Ambil level_name berdasarkan level_id yang sudah ada di database
        $levelName = DB::table('tbl_user_level')->where('id', $user->level_id)->value('level_name');

        // Ambil hanya kata pertama dari nama_lengkap
        $firstName = explode(' ', trim($validatedData['nama_lengkap'] ?? $user->nama_lengkap))[0];

        // Format AI Label baru
        $newAiLabel = "{$formattedId}-{$levelName}-{$firstName}-" . ($validatedData['nip'] ?? $user->nip);

        // Update data di database
        DB::table('tbl_user')->where('id', $user->id)->update(array_merge($validatedData, [
            'ai_label'   => $newAiLabel,
            'updated_at' => now('Asia/Jakarta'),
        ]));

        // **Jika AI Label sudah ada, update di API**
        if (! empty($oldAiLabel)) {
            $apiResponse = Http::withHeaders([
                'x-api-key' => '9115e6b4-f105-4d78-8bf3-718768d4eb48',
            ])->put("https://api-cpf.bantungoding.com/api/v1/recognition/subjects/{$oldAiLabel}", [
                'subject' => $newAiLabel,
            ]);

            if ($apiResponse->failed()) {
                return response()->json(['error' => 'Failed to update label in external API.'], $apiResponse->status());
            }
        } else {
            // Jika belum ada, tambahkan label baru ke API
            $apiResponse = Http::withHeaders([
                'x-api-key' => '9115e6b4-f105-4d78-8bf3-718768d4eb48',
            ])->post('https://api-cpf.bantungoding.com/api/v1/recognition/subjects', [
                'subject' => $newAiLabel,
            ]);

            if ($apiResponse->failed()) {
                return response()->json(['error' => 'Failed to add label in external API.'], $apiResponse->status());
            }
        }

        return response()->json(['message' => 'User updated successfully', 'ai_label' => $newAiLabel], 200);
    }
}